<?php
/**
 * @brief		Overview statistics extension: Downloads
 * @author		<a href='https://www.invisioncommunity.com'>Invision Power Services, Inc.</a>
 * @copyright	(c) Invision Power Services, Inc.
 * @license		https://www.invisioncommunity.com/legal/standards/
 * @package		Invision Community
 * @subpackage	Downloads
 * @since		16 Jan 2020
 */

namespace IPS\downloads\extensions\core\OverviewStatistics;

/* To prevent PHP errors (extending class does not exist) revealing path */

use IPS\DateTime;
use IPS\Db;
use IPS\Extensions\OverviewStatisticsAbstract;
use IPS\Theme;
use function defined;
use function is_array;

if ( !defined( '\IPS\SUITE_UNIQUE_KEY' ) )
{
	header( ( $_SERVER['SERVER_PROTOCOL'] ?? 'HTTP/1.0' ) . ' 403 Forbidden' );
	exit;
}

/**
 * @brief	Overview statistics extension: Downloads
 */
class Downloads extends OverviewStatisticsAbstract
{
	/**
	 * @brief	Which statistics page (activity or user)
	 */
	public string $page	= 'activity';

	/**
	 * Return the sub-block keys
	 *
	 * @note This is designed to allow one class to support multiple blocks, for instance using the ContentRouter to generate blocks.
	 * @return array
	 */
	public function getBlocks(): array
	{
		return array( 'downloads' );
	}

	/**
	 * Return block details (title and description)
	 *
	 * @param	string|NULL	$subBlock	The subblock we are loading as returned by getBlocks()
	 * @return	array
	 */
	public function getBlockDetails( string $subBlock = NULL ): array
	{
		/* Description can be null and will not be shown if so */
		return array( 'app' => 'downloads', 'title' => 'stats_overview_downloads', 'description' => 'stats_overview_downloads_desc', 'refresh' => 20 );
	}

	/** 
	 * Return the block HTML to show
	 *
	 * @param	array|string|null    $dateRange	String for a fixed time period in days, NULL for all time, or an array with 'start' and 'end' \IPS\DateTime objects to restrict to
	 * @param	string|NULL	$subBlock	The subblock we are loading as returned by getBlocks()
	 * @return	string
	 */
	public function getBlock( array|string $dateRange = NULL, string $subBlock = NULL ): string
	{
		$numbers = $this->getBlockNumbers( $dateRange, $subBlock );
		return Theme::i()->getTemplate( 'stats' )->overviewComparisonCount( $numbers['statsreports_current_count'], $numbers['statsreports_previous_count'] );
	}



	/**
	 * @param array|string  $dateRange=null
	 * @param string $subBlock = null
	 *
	 * @return array{statsreports_current_count: number|null, statsreports_previous_count: number|null}
	 */
	public function getBlockNumbers( array|string $dateRange = null, string $subBlock = null ) : array
	{
		$where			= NULL;
		$previousCount	= NULL;

		if( $dateRange !== NULL )
		{
			if( is_array( $dateRange ) )
			{
				$where = array(
					array( 'dtime > ?', $dateRange['start']->getTimestamp() ),
					array( 'dtime < ?', $dateRange['end']->getTimestamp() ),
				);
			}
			else
			{
				$currentDate	= new DateTime;
				$interval = static::getInterval( $dateRange );
				$initialTimestamp = $currentDate->sub( $interval )->getTimestamp();
				$where = array( array( 'dtime > ?', $initialTimestamp ) );

				$previousCount = Db::i()->select( 'COUNT(*)', 'downloads_downloads', array( array( 'dtime BETWEEN ? AND ?', $currentDate->sub( $interval )->getTimestamp(), $initialTimestamp ) ) )->first();
			}
		}

		$count = Db::i()->select( 'COUNT(*)', 'downloads_downloads', $where )->first();
		return [
			'statsreports_current_count' => $count,
			'statsreports_previous_count' => $previousCount,
		];
	}
}